# -*- coding: utf-8 -*-
# JadeDragon Technology (shengli.hu@yunside.com) 2018.07.06

import odoo
from functools import wraps
from odoo.sql_db import ConnectionPool, Connection
from odoo.sql_db import _logger, logging
from odoo.tools import config
import time
import urlparse
import psycopg2
from psycopg2.extensions import ISOLATION_LEVEL_AUTOCOMMIT
import re

re_from = re.compile('.* from "?([a-zA-Z_0-9]+)"? .*$')
re_into = re.compile('.* into "?([a-zA-Z_0-9]+)"? .*$')

# 多个只读连接池[(POOL,DSN)]
_ReadonlyPools = None
# 只读SQL计数器
readonly_sql_counter = 0


class Cursor(odoo.sql_db.Cursor):
    """
    实现数据库读写分离，负载轮询均衡
    读可以通过PG流复制的多个从Slave，建立连接池，按需取用
    这里需要在SQL中明确方言：/*JDG_READONLY*/，通常用在报表或某些性能查询中, 对于多维分析视图View, 通过名字来识别只读：jd_stat_view_
    需配置文件两个参数：
    readonly_db_uri = postgresql://username:passord@x.x.x.x:5555,postgresql://username:passord@x.x.x.x:5556
    readonly_db_maxconn = 64
    用户信息未填写则主库保持一致
    """

    def check(f):
        @wraps(f)
        def wrapper(self, *args, **kwargs):
            if self._closed:
                msg = 'Unable to use a closed cursor.'
                if self.__closer:
                    msg += ' It was closed at %s, line %s' % self.__closer
                raise psycopg2.OperationalError(msg)
            return f(self, *args, **kwargs)

        return wrapper

    @check
    def execute(self, query, params=None, log_exceptions=None):
        _logger.debug('JDG SQL DB')
        # 写操作启用事务，读操作不启用事务
        # TODO 需分析所有可能性，避免破坏事务完整性
        # if any([action in query for action in ['INSERT', 'UPDATE', 'CREATE', 'DELETE', 'ALTER', 'SAVEPOINT']]):
        #     if any([action in query for action in ['CREATE DATABASE', 'DROP DATABASE']]):
        #         self.autocommit(True)
        #     else:
        #         self.autocommit(False)
        # else:
        #     self.autocommit(True)
        # TODO log日志增加详细信息
        ################### 尝试使用只读集群查询 #######################
        if any([action in query for action in ['/*JDG_READONLY*/']]):
            # 频繁reset_view导致无法实时同步从库，暂时保持主库查询机制 'FROM "jd_stat_view_'
            if any([action in query for action in ['create or replace view']]):
                # 创建报表视图的时候不走只读
                return super(self.__class__, self).execute(query, params=params, log_exceptions=log_exceptions)
            # 执行均衡策略
            readonly_pool, readonly_dsn = readonly_next_pool(self.dbname)
            if readonly_pool is None:
                # 无池走正常逻辑
                return super(self.__class__, self).execute(query, params=params, log_exceptions=log_exceptions)
            # 获取只读连接
            readonly_cnx = readonly_pool.borrow(readonly_dsn)
            # 设置只读事务级别
            readonly_cnx.set_isolation_level(ISOLATION_LEVEL_AUTOCOMMIT)
            # 执行只读SQL
            if self.sql_log:
                now = time.time()
                _logger.debug("####POOL: %s \nDSN: %s \nQUERY: %s\n", readonly_pool, readonly_dsn, query)
            res = readonly_cnx.cursor().execute(query, params)

            # SQL高级日志
            if self.sql_log:
                delay = (time.time() - now) * 1E6

                res_from = re_from.match(query.lower())
                if res_from:
                    self.sql_from_log.setdefault(res_from.group(1), [0, 0])
                    self.sql_from_log[res_from.group(1)][0] += 1
                    self.sql_from_log[res_from.group(1)][1] += delay
                res_into = re_into.match(query.lower())
                if res_into:
                    self.sql_into_log.setdefault(res_into.group(1), [0, 0])
                    self.sql_into_log[res_into.group(1)][0] += 1
                    self.sql_into_log[res_into.group(1)][1] += delay

            # 只读查询计数器
            global readonly_sql_counter
            readonly_sql_counter += 1

            # 开启sql_log高级日志
            logging.debug("*********SQL BEGIN**: %s **SQL END**********" % query)
            self.print_log()

            # 释放连接到连接池
            chosen_template = config['db_template']
            templates_list = tuple(set(['template0', 'template1', 'postgres', chosen_template]))
            keep_in_pool = self.dbname not in templates_list
            readonly_pool.give_back(readonly_cnx, keep_in_pool=keep_in_pool)
            return res
        ##################### 尝试使用只读集群查询 ###########################

        return super(self.__class__, self).execute(query, params=params, log_exceptions=log_exceptions)


def readonly_next_pool(dbname):
    """
    自动轮询下一个连接池
    :return: Next Readonly_Pool
    """
    global readonly_sql_counter
    global _ReadonlyPools
    if _ReadonlyPools is None or len(_ReadonlyPools) == 0:
        init_readonly_pool(dbname)
    if _ReadonlyPools is None or len(_ReadonlyPools) == 0:
        # 无法初始化则返回None
        return None, None
    balance_count = len(_ReadonlyPools)
    next_pool = readonly_sql_counter % balance_count
    return _ReadonlyPools[next_pool]


def init_readonly_pool(dbname):
    """
    初始化只读连接池
    必须配置参数：
    readonly_db_uri = postgresql://x.x.x.x:5555,postgresql://x.x.x.x:5556
    readonly_db_maxconn = 64
    :return:
    """
    global _ReadonlyPools
    if _ReadonlyPools is None:
        _ReadonlyPools = []
        for dsn in readonly_dsn_list(dbname):
            maxconn = int(config['readonly_db_maxconn'] or 64)
            # 初始化连接池
            pool = ConnectionPool(maxconn)
            _ReadonlyPools.append((pool, dsn))
            # 初始化连接，充满
            cnx = []
            for i in range(0, maxconn):
                cnx.append(pool.borrow(dsn))
            for i in range(0, maxconn):
                pool.give_back(cnx[i], keep_in_pool=True)


def readonly_dsn_list(dbname):
    """
    读取多个只读连接配置
    :param dbname:
    :return: dsn连接参数[{'database':'','host':'','port':'','user':'','password':''}]
    """
    readonly_dsn_list = []
    if config.get('readonly_db_uri', False) and config.get('readonly_db_maxconn', False):
        for db_uri in config.get('readonly_db_uri').strip().split(','):
            p = urlparse.urlparse(db_uri)
            host, port, user, password = p.hostname, p.port, p.username, p.password
            if not (user or password):
                user, password = config['db_user'], config['db_password']
            dsn = {'database': dbname, 'host': p.hostname, 'port': p.port, 'user': user, 'password': password}
            readonly_dsn_list.append(dsn)

    return readonly_dsn_list


def readonly_close_all():
    """
    关闭所有只读连接, 清空连接池
    可用在系统关闭或者切库的时候
    @TODO
    :return:
    """
    global _ReadonlyPools
    if _ReadonlyPools:
        for rop, dsn in _ReadonlyPools:
            rop.close_all()
    _ReadonlyPools = None


def patch_sql_db():
    odoo.sql_db.Cursor = Cursor

#  vim:et:si:sta:ts=4:sts=4:sw=4:tw=79:
