# -*- coding: utf-8 -*-
import base64
import copy
import datetime
import os
import re
import time
import traceback
import uuid
from collections import defaultdict
from inspect import getmembers

import babel.dates
import pytz
import requests
import simplejson

import odoo
from odoo import models, api, fields, exceptions, SUPERUSER_ID
from odoo.exceptions import AccessError, MissingError, ValidationError, UserError
from odoo.models import _logger
from odoo.fields import Field, SpecialValue, FailedValue
from odoo.models import _normalize_ids
from odoo.osv import expression

from odoo.osv.query import Query
from odoo.tools import Collector
from odoo.tools import OrderedSet
from odoo.tools import config, DEFAULT_SERVER_DATETIME_FORMAT, DEFAULT_SERVER_DATE_FORMAT
from core.middleware import fields as jdg_fields
from odoo.tools.translate import _

os.environ['NLS_LANG'] = 'SIMPLIFIED CHINESE_CHINA.UTF8'

bucket = None

TYPE_OPERATION = {
    'read': u'读',
    'write': u'写',
    'unlink': u'删除',
    'create': u'创建'
}


def _format_row(row, model_obj):
    new_row = dict(row)
    for key in new_row.keys():
        field_model = model_obj._fields.get(key, None)
        if not field_model:
            continue
        if isinstance(new_row[key], datetime.datetime):
            str_date_time = new_row[key].strftime(DEFAULT_SERVER_DATETIME_FORMAT)
            new_row.update({key: str_date_time})
        elif isinstance(field_model, fields.Integer):
            tmp_val = row[key]
            new_row.update({key: int(tmp_val or 0)})
        elif isinstance(field_model, fields.Float):
            tmp_val = row[key]
            new_row.update({key: float(tmp_val or 0.0)})
        elif isinstance(field_model, jdg_fields.Many2oneChar):
            tmp_val = str(row[key] or '')
            if tmp_val:
                field_val = model_obj.env[field_model.comodel_name].browse(tmp_val)
                name_value = field_model.convert_to_read(field_val, model_obj)
                new_row.update({key: name_value})
            else:
                new_row.update({key: False})
        elif isinstance(field_model, fields.Many2one):
            tmp_val = int(row[key] or 0)
            if tmp_val:
                field_val = model_obj.env[field_model.comodel_name].browse(tmp_val)
                name_value = field_model.convert_to_read(field_val, model_obj)
                new_row.update({key: name_value})
            else:
                new_row.update({key: False})
    return new_row


def check(text):
    def decorator(func):
        def wrapper(self, *args, **kw):
            self.check_access_rights_button(text)
            return func(self, *args, **kw)
        return wrapper
    return decorator


class RemoteModel(models.AbstractModel):
    '''
    其他Odoo系统的模型
    '''
    _auto = False
    _register = False
    _abstract = False
    _transient = False
    _log_access = False
    _source_url = None
    _source_config = None
    _source_model = None
    _version = '10'

    @api.model
    def _setup_base(self, partial):
        """ Determine the inherited and custom fields of the model. """
        cls = type(self)
        if cls._setup_done:
            return

        # 1. determine the proper fields of the model: the fields defined on the
        # class and magic fields, not the inherited or custom ones
        cls0 = cls.pool.model_cache.get(cls._model_cache_key)
        if cls0 and cls0._model_cache_key == cls._model_cache_key:
            # cls0 is either a model class from another registry, or cls itself.
            # The point is that it has the same base classes. We retrieve stuff
            # from cls0 to optimize the setup of cls. cls0 is guaranteed to be
            # properly set up: registries are loaded under a global lock,
            # therefore two registries are never set up at the same time.

            # remove fields that are not proper to cls
            for name in set(cls._fields) - cls0._proper_fields:
                delattr(cls, name)
                cls._fields.pop(name, None)
            # collect proper fields on cls0, and add them on cls
            for name in cls0._proper_fields:
                field = cls0._fields[name]
                # regular fields are shared, while related fields are setup from scratch
                if not field.related:
                    self._add_field(name, field)
                else:
                    self._add_field(name, field.new(**field.args))
            cls._proper_fields = set(cls._fields)

        else:
            # retrieve fields from parent classes, and duplicate them on cls to
            # avoid clashes with inheritance between different models
            for name in cls._fields:
                delattr(cls, name)
            cls._fields = {}
            for name, field in getmembers(cls, Field.__instancecheck__):
                # do not retrieve magic, custom and inherited fields
                if not any(field.args.get(k) for k in ('automatic', 'manual', 'inherited')):
                    self._add_field(name, field.new())
            self._add_magic_fields()
            cls._proper_fields = set(cls._fields)

            cls.pool.model_cache[cls._model_cache_key] = cls
        self._add_remote_fields()
        # 2. add custom fields
        self._add_manual_fields(partial)

        # 3. make sure that parent models determine their own fields, then add
        # inherited fields to cls
        self._inherits_check()
        for parent in self._inherits:
            self.env[parent]._setup_base(partial)
        self._add_inherited_fields()

        # 4. initialize more field metadata
        cls._field_computed = {}  # fields computed with the same method
        cls._field_inverses = Collector()  # inverse fields for related fields
        cls._field_triggers = Collector()  # list of (field, path) to invalidate

        cls._setup_done = True

    @api.model_cr
    def _table_exist(self):
        return 1

    @api.model
    def _add_remote_fields(self):
        fields_list = []
        source_api = self._get_source_api_url()
        if not source_api:
            _logger.debug("Remote Model not Enabled, remote_base_data not Found.")
            return
        try:
            url = "%s/outer/web/dataset/model_fields" % source_api
            try:
                result = requests.get(url, data={'model': self._get_source_model()})
            except requests.ConnectionError:
                return _logger.error(u'远程模型通信初始化失败，url：%s' % url)
            fields_list = simplejson.loads(result.content)
        except Exception:
            _logger.error(
                'get remote model [%s] fields error, cause:%s' % (self._get_source_model(), traceback.format_exc()))
        for f in fields_list:  # TODO 远程的many2one, one2many要不要自动增加？
            f_type = f['type']
            if f_type == 'char':
                self._add_field(f['name'], fields.Char(string=f['string'], readonly=True))
            elif f_type == 'integer':
                self._add_field(f['name'], fields.Integer(string=f['string'], readonly=True))
            elif f_type == 'boolean':
                self._add_field(f['name'], fields.Boolean(string=f['string'], readonly=True))
            elif f_type == 'date':
                self._add_field(f['name'], fields.Date(string=f['string'], readonly=True))
            elif f_type == 'datetime':
                self._add_field(f['name'], fields.Datetime(string=f['string'], readonly=True))
            elif f_type == 'float':
                self._add_field(f['name'], fields.Float(string=f['string'], readonly=True))
            elif f_type == 'text':
                self._add_field(f['name'], fields.Text(string=f['string'], readonly=True))
            elif f_type == 'selection':
                self._add_field(f['name'], fields.Selection(string=f['string'], readonly=True))

    def _get_source_api_header(self):
        return {'Content-type': 'application/json', 'Accept': 'text/plain'}

    def _get_source_api_url(self):
        if self._name.startswith("remote."):
            # 模型名字: remote.source.res.users
            source = self._name.split(".")[1]
            source_url = config.get(source + ".url")
            return source_url

        if config.get(self._source_config, None):
            url = config.get(self._source_config)
            return url.strip()
        else:
            return self._source_url

    def _get_source_model(self):
        if self._name.startswith("remote."):
            # 模型名字: remote.source.res.users
            source_name = ".".join(self._name.split(".")[2:])
            return source_name
        if self._source_model:
            return self._source_model
        else:
            return self._name

    def json_call(self, method, args, kwargs, source_model=None):
        return self._json_call(method, args, kwargs, source_model)

    def _json_call(self, method, args, kwargs, source_model=None):
        host = self._get_source_api_url()
        if not host:
            _logger.error("remote base model not configuration.%s or %s, model:%s" % (
                self._source_url, self._source_config, self._name))
            return None
        url = '%s/outer/web/dataset/call_kw' % (self._get_source_api_url())
        if not source_model:
            remote_model = self._get_source_model()
        else:
            remote_model = source_model
        header = self._get_source_api_header()
        if not kwargs:
            kwargs = {}
        downstream_context = self.get_user_context(self.env.uid)
        app_key = self.get_app_key()
        downstream_context.update({
            'app_key': app_key
        })

        params = {
            'id': long(time.time() * 10000),
            'jsonrpc': '2.0',
            'method': 'call',
            'params': {
                'model': remote_model,
                'method': method,
                'args': args,
                'kwargs': kwargs,
                'downstream_context': downstream_context
            }
        }
        try:
            resp = requests.get(url,
                                json=params,
                                headers=header)
        except requests.ConnectionError:
            return _logger.error(u'远程模型调用失败，url：%s' % url)

        if resp.status_code == 200:
            result = resp.json()
            if 'error' in result.keys() and result['error']:
                raise exceptions.Warning('远程服务不可用。错误信息:%s' % str(result['error']))
            if result.has_key('result'):
                return result['result']
        else:
            raise exceptions.Warning('远程服务不可用。错误信息:%s' % resp.content)

    def get_app_key(self):
        '''
        获取远程模型的app_key
        :return:
        '''
        if self._name.startswith("remote."):
            # 模型名字: remote.source.res.users
            source = self._name.split(".")[1]
            app_key = config.get(source + ".app_key")
            return app_key
        return ""

    @api.multi
    def name_get(self):
        result = self._json_call('name_get', [self.ids, {}], {})
        return result

    @api.model
    def load_views(self, views, options=None):
        for item in views:
            item[0] = None
        if self.env.context and self.env.context.get('local', None):  # will load view from local
            return super(RemoteModel, self).load_views(views, options)

        kwargs = {
            'views': views,
            'options': options,
            'context': self.env.context
        }
        result = self._json_call('load_views', [], kwargs)
        local_name = self._name
        if hasattr(self, '_source_model'):
            remote_name = self._source_model
        else:
            remote_name = local_name
        form_fields_dict = result.get('fields_views', {}).get('form', {}).get('fields', {})
        list_fields_dict = result.get('fields_views', {}).get('list', {}).get('fields', {})
        for field_name in form_fields_dict:
            field_value = form_fields_dict[field_name]
            new_field_ctx = dict(field_value.get('context', {}))
            if field_value.get('relation', None):
                new_field_ctx.update({'remote_model': field_value.get('relation')})
                new_field_ctx.update({'parent_model': local_name})
            field_value.update({'context': new_field_ctx})
        for field_name in list_fields_dict:
            field_value = list_fields_dict[field_name]
            new_field_ctx = dict(field_value.get('context', {}))
            if field_value.get('relation', None):
                new_field_ctx.update({'remote_model': field_value.get('relation')})
                new_field_ctx.update({'parent_model': local_name})
            field_value.update({'context': new_field_ctx})
        return result

    @api.multi
    def get_formview_action(self):
        result = self._json_call('get_formview_action', [self.ids, {}], {})
        ctx = result.get('context', {})
        ctx.update({'remote_model': result['res_model'], 'local_model': self._name})
        result.update({'res_model': self._name, 'context': ctx})
        return result

    @api.model
    def fields_get(self, allfields=None, attributes=None):
        kwargs = {
            'allfields': allfields,
            'attributes': attributes
        }
        result = self._json_call('fields_get', [], kwargs)
        return result

    @api.model
    def fields_view_get(self, view_id=None, view_type='form', toolbar=False, submenu=False):
        if self.env.context.get('local'):
            return super(RemoteModel, self).fields_view_get(view_id, view_type, toolbar, submenu)
        kwargs = {
            'view_id': view_id,
            'view_type': view_type,
            'toolbar': toolbar,
            'submenu': submenu,
        }
        result = self._json_call('fields_view_get', [], kwargs)
        result.update({'model': self._name})
        return result

    @api.model
    def _search(self, args, offset=0, limit=None, order=None, count=False, access_rights_uid=None):
        kwargs = {
            'offset': offset,
            'limit': limit,
            'order': order,
            'count': count,
            'access_rights_uid': 1
        }
        result = self._json_call("_search", [args], kwargs)
        return result

    @api.model
    def _name_search(self, name='', args=None, operator='ilike', limit=100, name_get_uid=None):
        kwargs = {
            'args': args,
            'operator': operator,
            'limit': limit,
            'name_get_uid': 1
        }
        result = self._json_call("_name_search", [name], kwargs)
        return result

    @api.multi
    def read(self, fields=None, load='_classic_read'):
        result = self._json_call('read', [self.ids, fields], {})
        return result

    def _convert_to_cache(self, values, update=False, validate=True):
        """ Convert the ``values`` dictionary into cached values.

            :param update: whether the conversion is made for updating ``self``;
                this is necessary for interpreting the commands of *2many fields
            :param validate: whether values must be checked
        """
        fields = self._fields
        target = self if update else self.browse([], self._prefetch)
        result = {}
        for name, value in values.iteritems():
            new_value = value
            if isinstance(fields[name], jdg_fields.Many2oneChar) or isinstance(fields[name], odoo.fields.Many2one):
                if isinstance(value, list) and len(value) > 0:
                    new_value = value[0]
            result.update({
                name: fields[name].convert_to_cache(new_value, target, validate=validate)
            })
        return result

    @api.model
    def default_get(self, fields_list):
        result = self._json_call("default_get", [fields_list], {'context': self.env.context})
        return result

    @api.multi
    def write(self, vals):
        for key in vals.keys():
            field_model = self._fields.get(key)
            if field_model and isinstance(field_model, odoo.fields.One2many) and isinstance(vals[key], list):
                many_value = vals[key]
                count = len(many_value)
                for x in xrange(count - 1, -1, -1):
                    # 去掉删除所有的标记，因为在odoo8中，js端没有出来删除所有的行。
                    if isinstance(many_value[x], (tuple, list)) and many_value[x][0] == 5:  # delete all
                        del many_value[x]
        result = self._json_call('write', [self.ids, vals], {'context': self.env.context})
        return result

    @api.multi
    def unlink(self):
        result = self._json_call('unlink', [self.ids], {'context': self.env.context})
        return result

    @api.model
    def create(self, value):
        result = self._json_call('create', [value], {'context': self.env.context})
        record = self.browse(result)
        return record

    @api.multi
    def onchange(self, values, field_name, field_onchange):
        result = self._json_call('onchange', [self.ids, values, field_name, field_onchange],
                                 {'context': self.env.context})
        return result

    @api.multi
    def get_user_context(self, uid):
        '''
        获取用户的上下文
        '''
        context = {}
        user = ''
        company_id = 0
        # company_number = ''
        downstream_number = ''

        # partner_id = 0
        if uid:
            user_model = self.env.get('res.users')
            user_obj = user_model.browse(uid)
            user = user_obj.login
            if user_obj.company_id.id:
                company_id = user_obj.company_id.id
                # company_number = user_obj.company_id.number or ''
                downstream_number = 'number' in user_obj.company_id and user_obj.company_id.number or ''
                # if user_obj.company_id.store_id.id:
                #     store_id = user_obj.company_id.store_id.id
                #     store_number = user_obj.company_id.store_id.number or ''
        context.update({
            'user': user,
            'uid': uid if uid else 0,
            'company_id': company_id,
            # 'company_number': company_number,
            'downstream_number': downstream_number,
            'node_number': config.get('node_number', '-1'),
            # 'store_id': store_id
        })
        return context


def _convert_to_column(cls, value, record):
    '''
    为了支持oracle 的fid
    :param cls:
    :param value:
    :param record:
    :return:
    '''
    if not value:
        return 0
    try:
        return int(value)
    except Exception:
        return 0


def _add_custom_magic_fields(self):
    """
    除了增加自动字段外，再增加Many2one外部模型的名称字段。外部模型冗余id,name，方便在外部系统
    挂掉之后，还可以查看
    """

    def add(name, field):
        """ add ``field`` with the given ``name`` if it does not exist yet """
        if name not in self._fields:
            self._add_field(name, field)

    # cyclic import
    from odoo import fields

    # this field 'id' must override any other column or field
    self._add_field('id', fields.Id(automatic=True))

    add('display_name', fields.Char(string='Display Name', automatic=True,
                                    compute='_compute_display_name'))
    # 增加控制编辑按钮的字段
    if not (self._name.startswith("ir") or self._name.startswith('res')):
        add('hide_edit_button',
            fields.Boolean(string=u'显示编辑按钮', help=u'控制页面是否显示编辑按钮', compute='get_hide_edit_button', default=False,
                           store=False))

    if self._log_access:
        add('create_uid', fields.Many2one('res.users', string='Created by', automatic=True))
        add('create_date', fields.Datetime(string='Created on', automatic=True))
        add('write_uid', fields.Many2one('res.users', string='Last Updated by', automatic=True))
        add('write_date', fields.Datetime(string='Last Updated on', automatic=True))
        last_modified_name = 'compute_concurrency_field_with_access'
    else:
        last_modified_name = 'compute_concurrency_field'

    # this field must override any other column or field
    self._add_field(self.CONCURRENCY_CHECK_FIELD, fields.Datetime(
        string='Last Modified on', compute=last_modified_name, automatic=True))

    for fk in self._fields.keys():
        field_model = self._fields[fk]
        if isinstance(field_model, odoo.fields.Many2one) and hasattr(field_model, 'name_field'):
            name_field = getattr(field_model, 'name_field')
            if name_field:
                self._add_field(name_field, fields.Char(name_field))


@api.multi
def get_hide_edit_button(self):
    for each in self:
        each.hide_edit_button = False


@api.model
def _load_views(self, views, options=None):
    res = self._origin_load_views(views, options)
    if 'fields' in res:
        for key in res['fields'].keys():
            if not res['fields'][key]:
                continue
            if res['fields'][key].get('searchable', True) and key in self._fields and hasattr(self._fields[key],
                                                                                              'searchable'):
                res['fields'][key].update({'searchable': self._fields[key].searchable})
    return res


def _custom_write(self, values):
    '''
    冗余保存外部系统的name
    :param values:
    :return:
    '''
    # new_value = dict(values)
    for key in values:  # 保存图片到阿里云OSS
        if isinstance(self._fields[key], jdg_fields.AliOssImage):
            file_address = upload_to_alioss(values[key])
            values.update({key: file_address})

    for key in values.keys():
        field_model = self._fields.get(key, None)
        if field_model and isinstance(field_model, odoo.fields.Many2one) and hasattr(field_model, 'name_field'):
            name_field = getattr(field_model, 'name_field')
            if name_field:
                remote_obj = self.env[field_model._related_comodel_name].browse(values[key])
                name_value = remote_obj.name_get()
                if name_value:
                    for each_name_value in name_value:
                        name_text = each_name_value[1]
                        values.update({name_field: name_text})

        if field_model and isinstance(field_model, odoo.fields.Many2one) and isinstance(values[key], (tuple, list)):
            values[key] = values[key][0]  # 兼容odoo 8的 [id, name]格式
    result = self._origin_write_func(values)
    return result


def _custom_create(self, values):
    copy_value = dict(values)
    for key in values:  # 保存图片到阿里云OSS
        field_model = self._fields.get(key, None)
        if field_model and isinstance(field_model, jdg_fields.AliOssImage):
            file_address = upload_to_alioss(values[key])
            copy_value.update({key: file_address})

        if field_model and isinstance(field_model, odoo.fields.Many2one) and hasattr(field_model, 'name_field'):
            name_field = getattr(field_model, 'name_field')
            if name_field:
                remote_obj = self.env[field_model._related_comodel_name].browse(values[key])
                name_value = remote_obj.name_get()
                if name_value:
                    for each_name_value in name_value:
                        name_text = each_name_value[1]
                        copy_value.update({name_field: name_text})
        if field_model and isinstance(self._fields[key], fields.Many2one) and isinstance(values[key], (list, tuple)):
            if len(values[key]) == 2:  # odoo 8的 create方法，传值可能是{'product_id': [1,u'产品名称']}
                copy_value.update({key: values[key][0]})
    # _inherit_model加入到context中，给_inherits = {'xxx': 'xxx_id'}的情况下，给父类识别到子类。
    ctx = dict(self.env.context or {})
    ctx.update({'_inherit_model': self._name})
    result = self.with_context(ctx)._origin_create_func(copy_value)
    return result


def _custom_read(self, fields=None, load='_classic_read'):
    '''
    把name_field也读出来
    :param fields:
    :param load:
    :return:
    '''
    new_field_names = copy.copy(fields)
    has_name_field = False
    if fields:
        for fk in fields:
            field_model = self._fields.get(fk, None)
            if field_model and isinstance(field_model, odoo.fields.Many2one) and hasattr(field_model, 'name_field'):
                name_field = getattr(field_model, 'name_field')
                if name_field:
                    new_field_names.append(name_field)
                    has_name_field = True
    result = self._origin_read_func(new_field_names, load)
    return result


def _convert_to_read(self, value, record, user_name_get=True):
    '''
    对应外部模型，name_get不调用网络请求，直接从冗余字段来获取
    :param value:
    :param record:
    :param user_name_get:
    :return:
    '''
    if hasattr(self, 'name_field') and getattr(self, 'name_field'):
        if user_name_get and value:
            name_field = getattr(self, 'name_field')
            return [value.id, record[name_field]]
        else:
            return value.id
    else:
        return self._origin_field_convert_to_read(value, record, user_name_get)


def upload_to_alioss(file_base64):
    '''
    保存文件到阿里云，返回访问地址
    :param base64:
    :return:
    '''
    from odoo.tools import config
    import oss2

    endpoint = config.get('oss_endpoint', '')
    access_key_id = config.get('oss_access_key_id', '')
    access_key_secret = config.get('oss_access_key_secret', '')
    oss_bucket = config.get('oss_bucket', '')
    auth = oss2.Auth(access_key_id, access_key_secret)
    # bucket = oss2.Bucket(auth, endpoint, oss_bucket)

    if not endpoint:
        exceptions.Warning(u'没有配置oss_endpoint')
    if not access_key_id:
        exceptions.Warning(u'没有配置oss_access_key_id')
    if not access_key_secret:
        exceptions.Warning(u'没有配置oss_access_key_secret')
    if not oss_bucket:
        exceptions.Warning(u'没有配置oss_bucket')
    global bucket
    if not bucket:
        bucket = oss2.Bucket(auth, endpoint, config.get('oss_bucket', ''))
    file_name = "%s%d" % (str(uuid.uuid4()), long(time.time() * 100000))
    if file_base64:
        result = bucket.put_object(file_name, base64.b64decode(file_base64))
        if result and result.resp and result.resp.response:
            return result.resp.response.url
    return ''


@api.model
def _custom_read_group_format_result(self, data, annotated_groupbys, groupby, domain):
    """
    修改报表日期
    """

    sections = []
    for gb in annotated_groupbys:
        ftype = gb['type']
        value = data[gb['groupby']]

        # full domain for this groupby spec
        d = None
        if value:
            if ftype == 'many2one':
                value = value[0]
            elif ftype in ('date', 'datetime'):
                locale = self._context.get('lang') or 'en_US'
                fmt = DEFAULT_SERVER_DATETIME_FORMAT if ftype == 'datetime' else DEFAULT_SERVER_DATE_FORMAT
                tzinfo = None
                range_start = value
                range_end = value + gb['interval']
                # value from postgres is in local tz (so range is
                # considered in local tz e.g. "day" is [00:00, 00:00[
                # local rather than UTC which could be [11:00, 11:00]
                # local) but domain and raw value should be in UTC
                if gb['tz_convert']:
                    tzinfo = range_start.tzinfo
                    range_start = range_start.astimezone(pytz.utc)
                    range_end = range_end.astimezone(pytz.utc)

                range_start = range_start.strftime(fmt)
                range_end = range_end.strftime(fmt)
                if ftype == 'datetime':
                    label = babel.dates.format_datetime(
                        value, format='yyyy-MM-dd',
                        tzinfo=tzinfo, locale=locale
                    )
                else:
                    label = babel.dates.format_date(
                        value, format='yyyy-MM-dd',
                        locale=locale
                    )
                data[gb['groupby']] = ('%s/%s' % (range_start, range_end), label)
                d = [
                    '&',
                    (gb['field'], '>=', range_start),
                    (gb['field'], '<', range_end),
                ]

        if d is None:
            d = [(gb['field'], '=', value)]
        sections.append(d)
    sections.append(domain)

    data['__domain'] = expression.AND(sections)
    if len(groupby) - len(annotated_groupbys) >= 1:
        data['__context'] = {'group_by': groupby[len(annotated_groupbys):]}
    del data['id']
    return data


@api.model
def _customer_read_group_raw(self, domain, fields, groupby, offset=0, limit=None, orderby=False, lazy=True):
    self.check_access_rights('read')
    query = self._where_calc(domain)
    fields = fields or [f.name for f in self._fields.itervalues() if f.store]

    groupby = [groupby] if isinstance(groupby, basestring) else list(OrderedSet(groupby))
    groupby_list = groupby[:1] if lazy else groupby
    annotated_groupbys = [self._read_group_process_groupby(gb, query) for gb in groupby_list]
    groupby_fields = [g['field'] for g in annotated_groupbys]
    order = orderby or ','.join([g for g in groupby_list])
    groupby_dict = {gb['groupby']: gb for gb in annotated_groupbys}

    self._apply_ir_rules(query, 'read')
    for gb in groupby_fields:
        assert gb in fields, "Fields in 'groupby' must appear in the list of fields to read (perhaps it's missing in the list view?)"
        assert gb in self._fields, "Unknown field %r in 'groupby'" % gb
        gb_field = self._fields[gb].base_field
        assert gb_field.store and gb_field.column_type, "Fields in 'groupby' must be regular database-persisted fields (no function or related fields), or function fields with store=True"

    aggregated_fields = [
        f for f in fields
        if f != 'sequence'
        if f not in groupby_fields
        for field in [self._fields.get(f)]
        if field
        if field.group_operator
        if field.base_field.store and field.base_field.column_type
    ]

    field_formatter = lambda f: (
        self._fields[f].group_operator,
        self._inherits_join_calc(self._table, f, query),
        f,
    )
    select_terms = []
    divided_by_fields = []
    for f in aggregated_fields:
        f_result = field_formatter(f)
        tmp_filed = "%s(%s) AS %s" % f_result
        if f_result[0] == 'avg':
            divided_by = self._fields[f]._attrs.get('divided_by', None)
            if divided_by and len(divided_by) >= 2 and \
                    divided_by[0] in aggregated_fields and divided_by[1] in aggregated_fields:
                div_parent_formatter = field_formatter(divided_by[0])
                div_child_formatter = field_formatter(divided_by[1])
                div_parent = "%s(%s)" % (div_parent_formatter[0], div_parent_formatter[1])
                div_child = "%s(%s)" % (div_child_formatter[0], div_child_formatter[1])
                if len(divided_by) == 3 and divided_by[2] and isinstance(divided_by[2], (long, int, float)):
                    zoom = divided_by[2]
                    tmp_filed = "case when  abs(%s::INTEGER) = 0 then -1 else %s/%s*%f end AS %s" % (
                        div_child, div_parent, div_child, zoom, f_result[2])
                else:
                    tmp_filed = "case when abs(%s::INTEGER) = 0 then -1 else %s/%s end AS %s" % (
                        div_child, div_parent, div_child, f_result[2])
                divided_by_fields.append(div_child)
        select_terms.append(tmp_filed)
    # select_terms = ['%s(%s) AS "%s" ' % field_formatter(f) for f in aggregated_fields]

    for gb in annotated_groupbys:
        select_terms.append('%s as "%s" ' % (gb['qualified_field'], gb['groupby']))

    groupby_terms, orderby_terms = self._read_group_prepare(order, aggregated_fields, annotated_groupbys, query)
    from_clause, where_clause, where_clause_params = query.get_sql()
    if lazy and (len(groupby_fields) >= 2 or not self._context.get('group_by_no_leaf')):
        count_field = groupby_fields[0] if len(groupby_fields) >= 1 else '_'
    else:
        count_field = '_'
    count_field += '_count'

    prefix_terms = lambda prefix, terms: (prefix + " " + ",".join(terms)) if terms else ''
    prefix_term = lambda prefix, term: ('%s %s' % (prefix, term)) if term else ''
    having_fields = [item + " != 0" for item in divided_by_fields]
    divided_fields = ' and '.join(having_fields)
    if divided_fields:
        divided_fields = ' having ' + divided_fields
    query = """
        SELECT min("%(table)s".id) AS id, count("%(table)s".id) AS "%(count_field)s" %(extra_fields)s
        FROM %(from)s
        %(where)s
        %(groupby)s
        %(orderby)s
        %(limit)s
        %(offset)s
    """ % {
        'table': self._table,
        'count_field': count_field,
        'extra_fields': prefix_terms(',', select_terms),
        'from': from_clause,
        'where': prefix_term('WHERE', where_clause),
        'groupby': prefix_terms('GROUP BY', groupby_terms),
        'orderby': prefix_terms('ORDER BY', orderby_terms),
        'limit': prefix_term('LIMIT', int(limit) if limit else None),
        'offset': prefix_term('OFFSET', int(offset) if limit else None),
    }

    self._cr.execute(query, where_clause_params)
    fetched_data = self._cr.dictfetchall()

    if not groupby_fields:
        return fetched_data

    many2onefields = [gb['field'] for gb in annotated_groupbys if gb['type'] == 'many2one']
    if many2onefields:
        data_ids = [r['id'] for r in fetched_data]
        many2onefields = list(set(many2onefields))
        data_dict = {d['id']: d for d in self.browse(data_ids).read(many2onefields)}
        for d in fetched_data:
            d.update(data_dict[d['id']])

    data = map(lambda r: {k: self._read_group_prepare_data(k, v, groupby_dict) for k, v in r.iteritems()}, fetched_data)
    result = [self._read_group_format_result(d, annotated_groupbys, groupby, domain) for d in data]
    if lazy:
        # Right now, read_group only fill results in lazy mode (by default).
        # If you need to have the empty groups in 'eager' mode, then the
        # method _read_group_fill_results need to be completely reimplemented
        # in a sane way
        result = self._read_group_fill_results(
            domain, groupby_fields[0], groupby[len(annotated_groupbys):],
            aggregated_fields, count_field, result, read_group_order=order,
        )
    return result


def domain_to_sql(self, domain):
    if not domain:
        return ''
    if isinstance(domain, list):
        reversed_domain = reversed(domain)
        compare_list = []
        for each in reversed_domain:
            if each == '&':
                compare_list.append("&")
            elif each == '|':
                compare_list.append("|")
            elif isinstance(each, (tuple, list)) and len(each) == 3:
                field_name = each[0]
                operator = each[1].lower()
                compare_value = each[2]
                if field_name == 'id':
                    if hasattr(self, '_rec_id'):
                        field_name = self._rec_id
                if type(compare_value) in (
                        float, int, long):  # 不能用isinstanceof, 应用boolean也是isinstance(boolean,int)为True
                    tmp = "(%s %s %s)" % (field_name, operator, compare_value)
                    compare_list.append(tmp)
                elif isinstance(compare_value, bool) and isinstance(self._fields[field_name], fields.Boolean):
                    tmp = "(%s %s %s)" % (field_name, operator, 'TRUE' if compare_value else 'FALSE')
                    compare_list.append(tmp)
                elif isinstance(compare_value, bool) and operator == '=':
                    if compare_value:
                        tmp = "(%s is not null)" % (field_name,)
                        compare_list.append(tmp)
                    else:
                        tmp = "(%s is null)" % (field_name,)
                        compare_list.append(tmp)
                elif isinstance(compare_value, bool) and (operator == '!=' or operator == '<>'):
                    if compare_value:
                        tmp = "(%s is null)" % (field_name,)
                        compare_list.append(tmp)
                    else:
                        tmp = "(%s is not null)" % (field_name,)
                        compare_list.append(tmp)
                elif operator == 'ilike' or operator == 'like':
                    tmp = "(%s like '%s')" % (field_name, "%" + compare_value + "%")
                    compare_list.append(tmp)
                elif operator == 'in':
                    each_array = []
                    for each_compare_val in compare_value:
                        if isinstance(each_compare_val, (int, long, float)):
                            each_array.append(str(each_compare_val))
                        else:
                            each_array.append("'%s'" % each_compare_val)
                    # 去除sql条目限制
                    # if len(each_array) > 200:
                    #     each_array = each_array[:200]
                    each_in_str = ",".join(each_array)
                    tmp = "(%s in (%s))" % (field_name, each_in_str)
                    compare_list.append(tmp)
                else:
                    tmp = "(%s %s '%s')" % (field_name, operator, compare_value)
                    compare_list.append(tmp)
        final_sql = ' 1=1 '
        total_len = len(compare_list)
        for index in xrange(0, total_len, 1):
            current_domain = compare_list[index]
            next_domain = None
            if index < total_len - 1:
                next_domain = compare_list[index + 1]
            if not next_domain and current_domain != '|' and current_domain != '&':
                final_sql = " (%s and %s) " % (final_sql, current_domain)
            if next_domain == '|' and current_domain != '|' and current_domain != '&':
                final_sql = " (%s or %s) " % (final_sql, current_domain)
            elif current_domain != '|' and current_domain != '&':
                final_sql = " (%s and %s) " % (final_sql, current_domain)
        return final_sql
    else:
        return ''


@api.model
def get_min_max_value(self, field_name, domain):
    '''
    获取domain中的最小和最大值
    :param field_name: 要比较的字段
    :param domain:
    :return:[min_value,max_value]
    '''
    max_value = None
    min_value = None
    for item in domain:
        if len(item) == 3 and item[0] == field_name:
            if item[1] in (u'<', u'<='):
                max_value = item[2]
            elif item[1] in (u'>', u'>='):
                min_value = item[2]
            elif item[1] == u'=':
                min_value = max_value = item[2]
    return [min_value, max_value]

@api.model
def check_field_access_rights(self, operation, fields):
    """
    Check the user access rights on the given fields. This raises Access
    Denied if the user does not have the rights. Otherwise it returns the
    fields (as is if the fields is not falsy, or the readable/writable
    fields if fields is falsy).
    """
    if self._uid == SUPERUSER_ID:
        return fields or list(self._fields)

    def valid(fname):
        """ determine whether user has access to field ``fname`` """
        field = self._fields.get(fname)
        if field and field.groups:
            return self.user_has_groups(field.groups)
        else:
            return True

    if not fields:
        fields = filter(valid, self._fields)
    else:
        invalid_fields = set(filter(lambda name: not valid(name), fields))
        if invalid_fields:
            _logger.info('Access Denied by ACLs for operation: %s, uid: %s, model: %s, fields: %s',
                         operation, self._uid, self._name, ', '.join(invalid_fields))
            # raise AccessError(_('The requested operation cannot be completed due to security restrictions. '
            #                     'Please contact your system administrator.\n\n(Document type: %s, Operation: %s)') % \
            #                   (self._description, operation))
            raise AccessError(u'抱歉，您没有查看其他组织单据的权限：\r\n单据：%s, \r\n操作类型：%s，\r\n单据id:%s '
                              % (self._description, TYPE_OPERATION.get(operation), self.ids))

    return fields

@api.multi
def _read_from_database(self, field_names, inherited_field_names=[]):
    """ Read the given fields of the records in ``self`` from the database,
        and store them in cache. Access errors are also stored in cache.

        :param field_names: list of column names of model ``self``; all those
            fields are guaranteed to be read
        :param inherited_field_names: list of column names from parent
            models; some of those fields may not be read
    """
    env = self.env
    cr, user, context = env.args

    # make a query object for selecting ids, and apply security rules to it
    param_ids = object()
    query = Query(['"%s"' % self._table], ['"%s".id IN %%s' % self._table], [param_ids])
    self._apply_ir_rules(query, 'read')
    order_str = self._generate_order_by(None, query)

    # determine the fields that are stored as columns in tables;
    fields = map(self._fields.get, field_names + inherited_field_names)
    fields_pre = [
        field
        for field in fields
        if field.base_field.store and field.base_field.column_type
        if not (field.inherited and callable(field.base_field.translate))
    ]

    # the query may involve several tables: we need fully-qualified names
    def qualify(field):
        col = field.name
        res = self._inherits_join_calc(self._table, field.name, query)
        if field.type == 'binary' and (context.get('bin_size') or context.get('bin_size_' + col)):
            # PG 9.2 introduces conflicting pg_size_pretty(numeric) -> need ::cast
            res = 'pg_size_pretty(length(%s)::bigint)' % res
        return '%s as "%s"' % (res, col)

    qual_names = map(qualify, set(fields_pre + [self._fields['id']]))

    # determine the actual query to execute
    from_clause, where_clause, params = query.get_sql()
    query_str = """ SELECT %(qual_names)s FROM %(from_clause)s
                        WHERE %(where_clause)s %(order_str)s
                    """ % {
        'qual_names': ",".join(qual_names),
        'from_clause': from_clause,
        'where_clause': where_clause,
        'order_str': order_str,
    }

    result = []
    param_pos = params.index(param_ids)
    for sub_ids in cr.split_for_in_conditions(self.ids):
        params[param_pos] = tuple(sub_ids)
        cr.execute(query_str, params)
        result.extend(cr.dictfetchall())

    ids = [vals['id'] for vals in result]
    fetched = self.browse(ids)

    if ids:
        # translate the fields if necessary
        if context.get('lang'):
            for field in fields_pre:
                if not field.inherited and callable(field.translate):
                    f = field.name
                    translate = field.get_trans_func(fetched)
                    for vals in result:
                        vals[f] = translate(vals['id'], vals[f])

        # store result in cache for POST fields
        for vals in result:
            record = self.browse(vals['id'], self._prefetch)
            record._cache.update(record._convert_to_cache(vals, validate=False))

        # determine the fields that must be processed now;
        # for the sake of simplicity, we ignore inherited fields
        for f in field_names:
            field = self._fields[f]
            if not field.column_type:
                field.read(fetched)

    # Warn about deprecated fields now that fields_pre and fields_post are computed
    for f in field_names:
        field = self._fields[f]
        if field.deprecated:
            _logger.warning('Field %s is deprecated: %s', field, field.deprecated)

    # store failed values in cache for the records that could not be read
    missing = self - fetched
    if missing:
        extras = fetched - self
        if extras:
            raise AccessError(
                _(
                    "Database fetch misses ids ({}) and has extra ids ({}), may be caused by a type incoherence in a previous request").format(
                    ', '.join(map(repr, missing._ids)),
                    ', '.join(map(repr, extras._ids)),
                ))
        # mark non-existing records in missing
        forbidden = missing.exists()
        if forbidden:
            # store an access error exception in existing records
            # exc = AccessError(
            #     _('The requested operation cannot be completed due to security restrictions. Please contact your system administrator.\n\n(Document type: %s, Operation: %s)') % \
            #     (self._name, 'read')
            # )
            exc = AccessError(u'抱歉，您没有查看其他组织单据的权限：\r\n单据：%s, \r\n操作类型：%s，\r\n单据id: %s'
                              % (self._description if self._description else self._name, TYPE_OPERATION.get('read'), self.ids))
            forbidden._cache.update(FailedValue(exc))

@api.multi
def _check_record_rules_result_count(self, result_ids, operation):
    """ Verify the returned rows after applying record rules matches the
        length of ``self``, and raise an appropriate exception if it does not.
    """
    ids, result_ids = set(self.ids), set(result_ids)
    missing_ids = ids - result_ids
    if missing_ids:
        # Attempt to distinguish record rule restriction vs deleted records,
        # to provide a more specific error message
        self._cr.execute('SELECT id FROM %s WHERE id IN %%s' % self._table, (tuple(missing_ids),))
        forbidden_ids = [x[0] for x in self._cr.fetchall()]
        if forbidden_ids:
            # the missing ids are (at least partially) hidden by access rules
            if self._uid == SUPERUSER_ID:
                return
            _logger.info('Access Denied by record rules for operation: %s on record ids: %r, uid: %s, model: %s',
                         operation, forbidden_ids, self._uid, self._name)
            raise AccessError(u'抱歉，您没有查看其他组织单据的权限：\r\n单据：%s，\r\n操作类型：%s，\r\n单据id：%s'
                              % (self._description, TYPE_OPERATION.get(operation), forbidden_ids))
            # raise AccessError(_('The requested operation cannot be completed due to security restrictions. Please contact your system administrator.\n\n(Document type: %s, Operation: %s)') % \
            #                     (self._description, operation))
        else:
            # If we get here, the missing_ids are not in the database
            if operation in ('read', 'unlink'):
                # No need to warn about deleting an already deleted record.
                # And no error when reading a record that was deleted, to prevent spurious
                # errors for non-transactional search/read sequences coming from clients
                return
            _logger.info('Failed operation on deleted record(s): %s, uid: %s, model: %s', operation, self._uid,
                         self._name)
            raise MissingError(_('Missing document(s)') + ':' + _(
                'One of the documents you are trying to access has been deleted, please try again after refreshing.'))

from odoo.fields import Field


def patch_model():
    _origin_field_convert_to_read = fields.Many2one.convert_to_read
    _origin_write_func = models.BaseModel._write
    _origin_read_func = models.BaseModel.read

    _origin_create_func = models.BaseModel._create

    _origin_load_views = models.BaseModel.load_views

    fields.Integer.convert_to_column = _convert_to_column
    fields.Many2one._origin_field_convert_to_read = _origin_field_convert_to_read
    fields.Many2one.convert_to_read = _convert_to_read
    models.BaseModel._add_magic_fields = _add_custom_magic_fields
    models.BaseModel._write = _custom_write
    models.BaseModel._origin_write_func = _origin_write_func
    models.BaseModel._origin_read_func = _origin_read_func
    models.BaseModel.read = _custom_read
    models.BaseModel._origin_load_views = _origin_load_views
    models.BaseModel.load_views = _load_views
    models.BaseModel.domain_to_sql = domain_to_sql
    models.BaseModel.get_min_max_value = get_min_max_value
    models.BaseModel.get_hide_edit_button = get_hide_edit_button

    models.BaseModel._create = _custom_create
    models.BaseModel._origin_create_func = _origin_create_func
    models.BaseModel._read_group_raw = _customer_read_group_raw
    models.BaseModel._read_group_format_result = _custom_read_group_format_result

    # 重写权限提示方式
    models.BaseModel.check_field_access_rights = check_field_access_rights
    models.BaseModel._read_from_database = _read_from_database
    models.BaseModel._check_record_rules_result_count = _check_record_rules_result_count



