# -*- coding: utf-8 -*-
import os
import math
import random
import jinja2


from werkzeug.contrib.sessions import SessionStore

from odoo import http
from odoo.tools import config
from odoo.addons.web import controllers
from odoo.tools import ustr, consteq, frozendict
import psycopg2
import traceback

os.environ['NLS_LANG'] = 'SIMPLIFIED CHINESE_CHINA.UTF8'

oracle_pool = None
msserver_pool = None
bucket = None
# 全局REDIS连接
G_REDIS = None


class RedisSessionStore(SessionStore):
    """
    SessionStore that saves session to redis
    """

    def __init__(self, key_template='jdg-sess:%s', timeout=60 * 60 * 24, session_class=None):
        SessionStore.__init__(self, session_class=session_class)
        global G_REDIS
        self.redis = G_REDIS
        self.key_template = key_template
        self.timeout = timeout

    def get_session_key(self, sid):
        if isinstance(sid, unicode):
            sid = sid.encode('utf-8')
        return self.key_template % sid

    def save(self, session):
        from cPickle import dumps, HIGHEST_PROTOCOL

        key = self.get_session_key(session.sid)
        value = dumps(dict(session), HIGHEST_PROTOCOL)
        if self.redis.set(key, value):
            return self.redis.expire(key, self.timeout)

    def delete(self, session):
        key = self.get_session_key(session.sid)
        return self.redis.delete(key)

    def get(self, sid):

        if not self.is_valid_key(sid):
            return self.new()

        key = self.get_session_key(sid)
        saved = self.redis.get(key)
        self.redis.expire(key, self.timeout)
        from cPickle import loads

        if saved:
            data = loads(saved)
            return self.session_class(data, sid, False)
        else:
            return self.new()

    def list(self):
        """
        Lists all sessions in the store.
        """
        session_keys = self.redis.keys(self.key_template[:-2] + '*')
        return [s[len(self.key_template) - 2:] for s in session_keys]


@property
def _eas_cr(self):
    if hasattr(self, "_eas_cr") and getattr(self, "_eas_cr"):
        return self._eas_cr
    else:
        import cx_Oracle

        global oracle_pool
        if not oracle_pool:
            host = config.get("eas_db_host")
            user = config.get("eas_db_user")
            password = config.get("eas_db_password")
            service = config.get("eas_db_service")
            port = config.get("eas_db_port")
            pool_size = int(config.get('eas_db_pool_size', 50))
            dsn = "(DESCRIPTION=(ADDRESS=(PROTOCOL=TCP)(HOST=%(host)s)(PORT=%(port)s))(CONNECT_DATA=(SERVER=DEDICATED)(SERVICE_NAME=%(service)s)))" % {
                'host': host, 'port': port, 'service': service}
            oracle_pool = cx_Oracle.SessionPool(user=user, password=password,
                                                dsn=dsn,
                                                min=5, max=pool_size, increment=5)
        conn = oracle_pool.acquire()
        self._eas_conn = conn
        self._eas_cr = conn.cursor()
        return self._eas_cr


@property
def _io_cr(self):
    if hasattr(self, "_io_cr") and getattr(self, '_io_cr'):
        return self._io_cr
    else:
        import pymssql
        from DBUtils.PooledDB import PooledDB

        global msserver_pool
        host = config.get("io_db_host")
        user = config.get("io_db_user")
        password = config.get("io_db_password")
        database = config.get("io_db_database")
        pool_size = int(config.get('io_db_pool_size', 10))
        pool_max_connections = pool_size
        args = (0, 0, 0, pool_max_connections, 0, 0, None)
        conn_kwargs = {'host': host, 'user': user, 'password': password, 'database': database}
        if not msserver_pool:
            msserver_pool = PooledDB(pymssql, *args, **conn_kwargs)

        conn = msserver_pool.connection()
        self._io_conn = conn
        self._io_cr = conn.cursor(as_dict=True)
        return self._io_cr


origin_request_exit = http.WebRequest.__exit__


def _clean_conn(self):
    if hasattr(self, '_io_cr') and getattr(self, '_io_cr'):
        try:
            self._io_cr.close()
        except Exception:
            pass
        finally:
            self._io_cr = None
    if hasattr(self, '_eas_cr') and getattr(self, '_eas_cr'):
        try:
            self._eas_cr.close()
        except Exception:
            pass
        finally:
            self._eas_cr = None
    if hasattr(self, '_io_conn') and getattr(self, '_io_conn'):
        try:
            self._io_conn.commit()
            self._io_conn.close()
        except Exception:
            pass
        finally:
            self._io_conn = None
    if hasattr(self, '_eas_conn') and getattr(self, '_eas_conn'):
        try:
            if oracle_pool:
                oracle_pool.release(self._eas_conn)
                self._eas_conn = None
        except Exception:
            pass
        finally:
            self._eas_conn = None


def __request_exit__(self, exc_type, exc_value, traceback):
    origin_request_exit(self, exc_type, exc_value, traceback)
    _clean_conn(self)


# origin_base_model_del = odoo.models.BaseModel.__del__


def __del_base_model__(self):
    _clean_conn(self)


# origin_environment_del = odoo.api.Environment.__del__


def __del_environment__(self):
    _clean_conn(self)


import odoo
from odoo.http import OpenERPSession
from odoo.tools.func import lazy_property


@lazy_property
def session_store(self):
    return RedisSessionStore(session_class=OpenERPSession)


def session_gc(session_store):
    pass


def _call_kw_ex(self, model, method, args, kwargs):
    from odoo.api import call_kw
    from odoo.models import check_method_name
    from odoo.http import request
    check_method_name(method)
    model_obj = None
    remote_model = None
    parent_model = None
    if kwargs and kwargs.get('context', None):
        context = kwargs.get('context')
        remote_model = context.get('remote_model', None)
        parent_model = context.get('parent_model', None)

    if remote_model and parent_model:
        for key in request.env.keys():
            m = request.env[key]
            # 如果这是一个远程模型的字段，从request中取出与该模型相对应的model_obj
            if hasattr(m, '_source_model') == remote_model:
                model_obj = m
                break
        if model_obj:
            return call_kw(model_obj, method, args, kwargs)
        else:
            result = request.env[parent_model].json_call(method, args, kwargs, remote_model)
            return result


    try:
        model_obj = request.env[model]
    except:
        pass
    if model_obj != None:
        return call_kw(model_obj, method, args, kwargs)
    else:
        for key in request.env.keys():
            m = request.env[key]
            if hasattr(m, '_source_model') and hasattr(m, '_name'):
                if m._source_model == model:
                    model = m._name
                    break
        return call_kw(request.env[model], method, args, kwargs)

# When running on compiled windows binary, we don't have access to package loader.
# 更改数据库页面为/jd_init/views/database_manager.html
# 添加数据库主页logo随机数
path = os.path.realpath(os.path.join(os.path.dirname(__file__), '..', '..', 'base/app_base/jd_init/views'))
loader = jinja2.FileSystemLoader(path)
env = jinja2.Environment(loader=loader, autoescape=True)


db_monodb = http.db_monodb
DBNAME_PATTERN = '^[a-zA-Z0-9][a-zA-Z0-9_.-]+$'


def _render_template(self, **d):
    d.setdefault('manage',True)
    d['insecure'] = odoo.tools.config['admin_passwd'] == 'admin'
    d['list_db'] = odoo.tools.config['list_db']
    d['langs'] = odoo.service.db.exp_list_lang()
    d['countries'] = odoo.service.db.exp_list_countries()
    d['pattern'] = DBNAME_PATTERN
    # databases list
    d['databases'] = []
    #添加随机数
    d['random_number'] = str(int(math.ceil(random.random() * 100000)))
    try:
        d['databases'] = http.db_list()
    except odoo.exceptions.AccessDenied:
        monodb = db_monodb()
        if monodb:
            d['databases'] = [monodb]
    return env.get_template("database_manager.html").render(d)


def serialize_exception(e):
    tmp = {
        "name": type(e).__module__ + "." + type(e).__name__ if type(e).__module__ else type(e).__name__,
        "debug": traceback.format_exc(),
        "message": ustr(e),
        "arguments": to_jsonable(e.args),
        "exception_type": "internal_error"
    }
    if isinstance(e, odoo.exceptions.UserError):
        tmp["exception_type"] = "user_error"
    elif isinstance(e, odoo.exceptions.Warning):
        tmp["exception_type"] = "warning"
    elif isinstance(e, odoo.exceptions.RedirectWarning):
        tmp["exception_type"] = "warning"
    elif isinstance(e, odoo.exceptions.AccessError):
        tmp["exception_type"] = "access_error"
    elif isinstance(e, odoo.exceptions.MissingError):
        tmp["exception_type"] = "missing_error"
    elif isinstance(e, odoo.exceptions.AccessDenied):
        tmp["exception_type"] = "access_denied"
    elif isinstance(e, odoo.exceptions.ValidationError):
        tmp["exception_type"] = "validation_error"
    elif isinstance(e, odoo.exceptions.except_orm):
        tmp["exception_type"] = "except_orm"
    # 数据库错误提示友好
    elif isinstance(e, psycopg2.Error):
        tmp["debug"] = "数据库操作错误，请联系系统管理员\r\n\r\n详情请查看日志\r\n%s" % ustr(e)
        tmp["is_pg_error"] = True
        tmp['stack'] = traceback.format_exc()
    # 其他错误统一提示
    else:
        tmp["debug"] = "服务器内部错误，请联系系统管理员\r\n%s" % ustr(e)
        tmp["is_pg_error"] = True
        tmp['stack'] = traceback.format_exc()
    return tmp


def to_jsonable(o):
    if isinstance(o, str) or isinstance(o,unicode) or isinstance(o, int) or isinstance(o, long) \
        or isinstance(o, bool) or o is None or isinstance(o, float):
        return o
    if isinstance(o, list) or isinstance(o, tuple):
        return [to_jsonable(x) for x in o]
    if isinstance(o, dict):
        tmp = {}
        for k, v in o.items():
            tmp[u"%s" % k] = to_jsonable(v)
        return tmp
    return ustr(o)

def patch_http():

    http.WebRequest.eas_cr = _eas_cr
    http.WebRequest.io_cr = _io_cr
    odoo.models.BaseModel.eas_cr = _eas_cr
    odoo.api.Environment.eas_cr = _eas_cr
    odoo.models.BaseModel.io_cr = _io_cr
    odoo.api.Environment.io_cr = _io_cr
    http.serialize_exception = serialize_exception

    http.WebRequest.__exit__ = __request_exit__
    odoo.models.BaseModel.__del__ = __del_base_model__
    odoo.api.Environment.__del__ = __del_environment__
    # 配置系统参数
    import sys
    odoo.tools.config.parse_config(sys.argv[1:])
    # 是否开启redis
    if config.get('redis_session', False):
        if config.get('redis_host', None) and config.get('redis_port', None) and config.get('redis_db', None):
            odoo.http.Root.session_store = session_store
            odoo.http.session_gc = session_gc
            import redis

            global G_REDIS
            print 'use redis session store.'
            G_REDIS = redis.StrictRedis(host=config['redis_host'], port=config['redis_port'], db=config['redis_db'])
            odoo.http.redis_ref = G_REDIS
        else:
            print 'redis_host, redis_port, redis_db is not exist in configuration file, use default session store.'

    else:
        print 'use default session store'
    controllers.main.Database._render_template = _render_template
    controllers.main.DataSet._call_kw = _call_kw_ex