# -*- encoding: utf-8 -*-
# 项目：JDG-
# 模块名称：
# 描述：
# Copyright 2018 JDG <www.yunside.com>
# Created by wxh (xianhuo.weng@yunside.com) at  2019/11/19 - 9:07 AM

from odoo import fields, api, models
from core.middleware.utils import today
import datetime
from odoo.exceptions import ValidationError


class TodoOverdue(models.Model):
    _name = 'jd.todo.overdue'
    _inherit = 'jdg.abstract.base.biz'
    _description = u'逾期记录'
    _seq_prefix = 'TO'

    """
    唯一标识：关联待办 + 逾期人 + 待办开始日期
    """

    todo_id = fields.Many2one('jd.todo.task', string=u'关联待办', required=True)
    user_id = fields.Many2one('res.users', string=u'逾期人', required=True)
    title = fields.Char(string=u'待办标题')
    content = fields.Char(string=u'待办内容')
    identity = fields.Char(string=u'待办标识')

    state = fields.Selection(compute='_get_state', store=True)
    # 待办带出的信息
    days_due = fields.Integer(string=u'逾期天数', default=1, compute='_get_todo_info', store=True,
                              help=u'待办应在逾期日期前完成。待办开始日期 + 逾期天数 = 逾期日期')
    priority = fields.Integer(string=u'优先级', default=100, compute='_get_todo_info', store=True, help=u'数值越小代表优先级越高')

    date_start = fields.Date(string=u'待办开始日期', default=lambda self: fields.datetime.now())
    date_finish = fields.Date(string=u'完成日期', default=False)

    date_due = fields.Date(string=u'待办逾期日期', compute='_get_due_info', store=True, help=u'待办任务应在待办逾期日期前完成')
    days_overdue = fields.Integer(string=u'已逾期天数', compute='_get_due_info', store=True)

    @api.multi
    @api.depends('state')
    def get_hide_edit_button(self):
        """
        编辑按钮可见性：根据状态控制编辑按钮可见性。子类中可以覆盖该方法添加更详细的控制条件。
        :return:
        """
        for each in self:
            each.hide_edit_button = each.state in ('audit', 'cancel')
            
    @api.model
    def create(self, vals):
        # 创建即为确认
        vals.update({'state': 'confirm'})
        return super(TodoOverdue, self).create(vals)
    
    @api.multi
    @api.depends('date_finish')
    def _get_state(self):
        for item in self:
            # 若完成日期不为空则状态置为审核
            item.state = 'audit' if item.date_finish else 'confirm'
    
    @api.multi
    @api.depends('todo_id')
    def _get_todo_info(self):
        for item in self:
            if item.todo_id:
                item.days_due = item.todo_id.days_due
                item.priority = item.todo_id.priority
    
    @api.multi
    @api.depends('days_due', 'date_start', 'date_finish')
    def _get_due_info(self):
        for item in self:
            date_may_finish = item.date_finish and datetime.datetime.strptime(item.date_finish, '%Y-%m-%d') or datetime.datetime.now()
            date_due = (datetime.datetime.strptime(item.date_start, "%Y-%m-%d") + datetime.timedelta(
                days=item.days_due))

            # 计算超期天数
            days_overdue = (date_may_finish - date_due).days + 1
            days_overdue = days_overdue if days_overdue > 0 else 0

            item.days_overdue = days_overdue
            item.date_due = date_due.strftime("%Y-%m-%d")
