# -*- coding:utf-8 -*-
# Copyright 2018 JDG <www.yunside.com>
# Created by bjccdsrlcr (longjie.jiang@yunside.com) @ 2018/5/8
import logging
import odoorpc
from odoo import models, fields, api
from odoo.exceptions import ValidationError

_logger = logging.getLogger(__name__)

PRODUCT_MAPPING = {}  # {'remote_id-node_id': 'local_product'}


class DataNode(models.Model):
    """
    数据节点
    """
    _name = 'jd.data.node'
    _inherit = 'jdg.abstract.base.data.m'
    _description = u"数据节点"
    _order = 'number asc'

    # 肉猪养殖节点
    NODE_FARM_MAPPING = {
        '127.0.0.1': {
            'store_org_xml_id': 'jd_bc_master.store_org_01',
            'company_id': 'jd_bc_master.company_001',
        },
        'jd-gz-01.yunside.com': {
            'store_org_xml_id': 'jd_bc_master.store_org_01',
            'company_id': 'jd_bc_master.company_001',
        },

    }
    # 种猪生产系统
    BREEDING_PRODUCTION_MAPPING = {
        '127.0.0.1': {
            'store_org_xml_id': 'jd_bc_master.store_org_01',
            'company_id': 'jd_bc_master.company_001',
        },
        'jd-gz-01.yunside.com': {
            'store_org_xml_id': 'jd_bc_master.store_org_01',
            'company_id': 'jd_bc_master.company_001',
        },

    }

    node_url = fields.Char(u'URL', required=True)
    login = fields.Char(string=u'用户名', index=True, default='admin')
    password = fields.Char(string=u'密码', index=True)
    port = fields.Char(string=u'端口', index=True, default='80')
    db = fields.Char(string=u'数据库', index=True, required=True)
    data_node_sync_ids = fields.One2many('jd.data.node.sync', 'data_node_id', string=u'同步记录')
    download_history = fields.One2many('jd.download.history', compute='get_download_history', string=u'下发记录')

    @api.multi
    @api.depends('state')
    def get_hide_edit_button(self):
        for item in self:
            item.hide_edit_button = item.state in ('enable', 'discard')

    @api.multi
    def get_download_history(self):
        history_model = self.env['jd.download.history'].sudo()
        for record in self:
            history = history_model.search([('data_node_id', '=', record.id)], order='id desc')
            record.download_history = [item.id for item in history]

    @api.multi
    @api.constrains('number', 'name')
    def _check_number(self):
        """
        编码合法性校验, 覆盖忽略
        """
        pass

    def _remote_login(self):
        """
        登录当前节点
        :return:
        """
        # odoo = odoorpc.ODOO(host=self.node_url, port=self.port, timeout=600)
        try:
            odoo = odoorpc.ODOO(host=self.node_url, port=self.port, timeout=600)
            odoo.login(db=self.db, login=self.login, password=self.password)
            return odoo
        except Exception as e:
            raise ValidationError('远程服务不可用!请检查下游服务器连接')

    @api.model
    def _login_node(self, node):
        """
        登录某个节点
        by Shengli
        """
        try:
            odoo = odoorpc.ODOO(host=node['node_url'], port=node['port'], timeout=600)
            odoo.login(db=node['db'], login=node['login'], password=node['password'])
            return odoo
        except Exception as e:
            _logger.error('login to :%s fail' % node['node_url'])
            raise ValidationError('远程服务不可用!请检查下游服务器连接')

    @api.multi
    def _get_mapping_company(self):
        """
        获取节点的公司
        :return:
        """
        self.ensure_one()
        xml_id = DataNode.NODE_FARM_MAPPING.get(self[0].node_url)
        return self.env.ref(xml_id['company_id'])

    @api.model
    def convert_remote_to_local_id_by_node_number(self, data_node_number, data_type, data_src_id):
        sql = '''
            select
            center_id
            from jd_m_data_node_sync ns
            inner join jd_m_data_node n on n.id=ns.data_node_id
            where ns.data_type=%s
            and ns.data_src_id=%s
            and n.number=%s
            order by ns.id ASC
            limit 1
        '''
        self.env.cr.execute(sql, (data_type, data_src_id, data_node_number))
        rs = self.env.cr.dictfetchone()
        if rs:
            return rs['center_id']
        return None

    @api.model
    def convert_remote_to_local_id(self, data_node_id, data_type, data_src_id):
        """
        远程记录ID，转换为本地记录ID
        :param data_node_id: 数据节点记录ID
        :param data_type: 本地数据模型名称
        :param data_src_id: 远程记录ID
        :return:
        """
        sql = '''
            select
            center_id
            from jd_m_data_node_sync
            where data_type=%s
            and data_src_id=%s
            and data_node_id=%s
            order by id ASC
            limit 1
        '''
        self.env.cr.execute(sql, (data_type, data_src_id, data_node_id))
        rs = self.env.cr.dictfetchone()
        if rs:
            return rs['center_id']
        return None


class DataNodeDataSync(models.Model):
    _name = 'jd.data.node.sync'
    _description = u"数据节点同步记录"
    _order = 'data_type desc, id desc'

    data_node_id = fields.Many2one('jd.data.node', string=u'数据节点', required=True, ondelete='restrict')
    data_type = fields.Char(string=u'数据源模型', required=True, index=True)
    # 一条数据的id
    data_src_id = fields.Integer(string=u'数据源ID', index=True)
    # name字段
    data_src_name = fields.Char(string=u'数据源名称', required=True, index=True)
    data_src_number = fields.Char(string=u'数据源编码', index=True)
    center_id = fields.Integer(string=u'中心ID', index=True)
    state = fields.Selection([('synced', '已同步'), ('confirmed', '已确认')], string=u'状态', index=True, required=True)

    _sql_constraints = [
        ('ids_uniq',
         'unique (data_node_id,data_type,data_src_id,center_id)',
         u'(data_node_id,data_type,data_src_id,center_id)组合不能重复!'),
    ]
