# -*- coding: utf-8 -*-
# Copyright 2018 JDG <www.yunside.com>
# Created by summerrain <summerrain5445@gmail.com> at 2018/5/31

from odoo import models, fields, api
from odoo.exceptions import ValidationError
from odoo.osv import expression

STOCK_STATE = [('draft', u'新建'),
               ('confirm', u'确认'),
               ('correction', u'修正'),
               ('closed', u'关账'),
               ('cancel', u'作废')]


class AbstractBaseBizStock(models.AbstractModel):
    _name = 'jdg.abstract.base.biz.stock'
    _inherit = 'jdg.abstract.base.biz.fin'
    _description = u"库存单据基类"
    _order = 'number desc'

    state = fields.Selection(STOCK_STATE, string=u'状态', default='draft', help=u'控制数据是否有效')
    origin = fields.Reference(string=u'来源', selection='_reference_models', readonly=True)
    lot = fields.Char(String=u'批次')

    time_correction = fields.Datetime(string=u'修正时间')
    uid_correction = fields.Many2one('res.users', string=u'修正者', readonly=True, help=u'由修正动作自动反写')


    @api.model
    def _reference_models(self):
        models = self.env['ir.model'].search([('state', '!=', 'manual')])
        return [(model.model, model.name)
                for model in models
                if not model.model.startswith('ir.')]

    @api.multi
    def do_confirm(self):
        """
        确认动作，改变单据状态，子类覆盖该方法时注意逻辑
        :return:
        """
        for record in self:
            if record.state == 'draft' or record.state == 'correction':
                record.jdg_sudo().write({
                    'state': 'confirm',
                    'uid_confirm': self.env.user.id,
                    'time_confirm': fields.Datetime.now()
                })
                record.get_is_post()
            else:
                raise ValidationError('只有新建或修正状态下的单据才能确认')

    @api.multi
    def do_correction(self):
        """
        修正动作，改变单据状态。
        :return:
        """
        for record in self:
            if record.state == 'confirm':
                record.jdg_sudo().write({
                    'state': 'correction',
                    'uid_correction': self.env.user.id,
                    'time_correction': fields.Datetime.now()
                })
                record.get_no_post()
            else:
                raise ValidationError('只有确认状态下的单据才能修正')

    @api.multi
    def do_cancel(self):
        """
        作废动作，改变单据状态，子类覆盖该方法时注意逻辑
        :return:
        """
        for record in self:
            if record.state in ('confirm', 'correction'):
                record.jdg_sudo().write({
                    'state': 'cancel',
                    'uid_cancel': self.env.user.id,
                    'time_cancel': fields.Datetime.now()
                })
                record.get_no_post()
            else:
                raise ValidationError('只有确认和修正状态下的单据才能作废')

    @api.multi
    @api.depends('state')
    def get_hide_edit_button(self):
        """
        编辑按钮可见性：根据状态控制编辑按钮可见性。子类中可以覆盖该方法添加更详细的控制条件。
        :return:
        """
        for each in self:
            each.hide_edit_button = True if each.state in ('confirm', 'cancel', 'closed') else False