package log;

import log.data.HdBdBsBatchRelationLog;

import java.io.*;
import java.util.*;

/**
 * 用来检查批次存栏流水中哪些猪进出流水不相等
 * 从文件读入，每行为一个数据
 * 每行格式为：编号 耳号 进/出
 * 使用空格隔开
 *
 * @author 00137030
 * @date 2022/08/07
 */
public class HdBdBsBatchRelationLogCheck {

    /**
     * 状态为“进”总数
     */
    private static Integer inNumber = 0;
    /**
     * 状态为“出”总数
     */
    private static Integer outNumber = 0;
    /**
     * 状态为 “FALSE”/空 总数
     */
    private static Integer errorNumber = 0;
    /**
     * “进 - 出 = 1”数量
     */
    private static Integer in = 0;
    /**
     * “进 - 出 > 1”数量
     */
    private static Integer inError = 0;
    /**
     * “进 < 出”数量
     */
    private static Integer outError = 0;
    /**
     * “进 == 出”但存在状态为 “FALSE”/空 数量
     */
    private static Integer stateError = 0;

    /**
     * 从文件读入数据保存到 ArrayList
     *
     * @param fileInPath 输入文件路径
     * @return 保存了批次存栏流水信息的 ArrayList
     */
    private static ArrayList<HdBdBsBatchRelationLog> readFile(String fileInPath) {
        // 暂存一行字符
        String hdBdBsBatchRelationLog;
        // 创建 ArrayList ，用来保存每行数据所创建的对象
        ArrayList<HdBdBsBatchRelationLog> hdBdBsBatchRelationLogList = new ArrayList<>();
        try {
            // 文件字符输入流，可能存在 FileNotFoundException
            FileReader fileReader = new FileReader(fileInPath);
            // 缓冲区，有 readLine() 函数，方便使用
            BufferedReader bufferedReader = new BufferedReader(fileReader);
            // 按行从文件读入，可能存在 IOException
            while ((hdBdBsBatchRelationLog = bufferedReader.readLine()) != null) {
                // 空字符的行不做处理，包括 [" ", "\t", "\r", "\n"]
                if ((hdBdBsBatchRelationLog.replaceAll("\\s+", "")).length() > 0) {
                    // 去除开头和结尾的空白字符后，按空白字符切割字符串保存到字符数组中
                    String[] logStrings = hdBdBsBatchRelationLog.trim().split("\\s+");
                    // 每一行数据创建一个对象保存到 ArrayList 中
                    if (logStrings.length == 3) {
                        // 导出 Excel 表中，若状态为空，默认值为“FALSE”
                        hdBdBsBatchRelationLogList.add(new HdBdBsBatchRelationLog(logStrings[0], logStrings[1], logStrings[2]));
                    } else if (logStrings.length == 2) {
                        // 网页中，若状态为空，则无值
                        hdBdBsBatchRelationLogList.add(new HdBdBsBatchRelationLog(logStrings[0], logStrings[1], null));
                    } else {
                        throw new RuntimeException("请检查输入数据格式：" + hdBdBsBatchRelationLog);
                    }
                }
            }
            // 关闭缓冲
            bufferedReader.close();
            // 关闭文件字符输入流
            fileReader.close();
        } catch (IOException e) {
            // 捕获可能出现 FileNotFoundException 和 IOException
            e.printStackTrace();
        }
        // 返回值未保存了批次存栏流水信息的 ArrayList
        return hdBdBsBatchRelationLogList;
    }

    /**
     * 将数据写入文件
     *
     * @param checkResult 需要输出的结果集合
     * @param fileOutPath 输出文件路径
     */
    private static void writeFile(ArrayList<HashMap<String, ArrayList<Integer>>> checkResult, String fileOutPath) {
        LinkedHashSet<String> resultSet = new LinkedHashSet<>();
        for (HashMap<String, ArrayList<Integer>> map : checkResult) {
            for (Map.Entry<String, ArrayList<Integer>> entry : map.entrySet()) {
                // 使用 LinkedHashSet 去重，并保留顺序
                resultSet.add(entry.getKey());
            }
        }
        int count = 0;
        try {
            // 文件字符输出流
            FileWriter fileWriter = new FileWriter(fileOutPath);
            // 缓冲区，可以一次写入一行，方便使用
            BufferedWriter bufferedWriter = new BufferedWriter(fileWriter);
            for (String breedingId : resultSet) {
                // 写入耳号
                bufferedWriter.write(breedingId);
                // 换行
                if (++count < resultSet.size()) {
                    bufferedWriter.newLine();
                }
                // 刷新缓冲区
                bufferedWriter.flush();
            }
            // 关闭缓冲
            bufferedWriter.close();
            // 关闭文件字符输出流
            fileWriter.close();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * 检查输入文件中的编号是否存在相同的情况，避免复制时出错影响检查结果
     *
     * @param logList 从文件读入的数据
     * @return 编号是否不重复
     */
    private static boolean numberCheck(ArrayList<HdBdBsBatchRelationLog> logList) {
        HashSet<String> hashSet = new HashSet<>();
        // 遍历 ArrayList 并往 HashSet 添加元素
        for (HdBdBsBatchRelationLog log : logList) {
            hashSet.add(log.getNumber());
        }
        // Set 中不能保存相同的元素，对比 ArrayList 和 HashSet 的数据的数量即可判断 ArrayList 中是否有重复数据
        return hashSet.size() == logList.size();
    }

    /**
     * 对批次存栏流水信息进行检查
     *
     * @param hdBdBsBatchRelationLogList 保存了批次存栏流水信息的 ArrayList
     * @return 返回不正常的猪只的耳号，若全部正常则返回 null
     */
    private static ArrayList<HashMap<String, ArrayList<Integer>>> hdBdBsBatchRelationLogCheck(ArrayList<HdBdBsBatchRelationLog> hdBdBsBatchRelationLogList) {
        // 使用 HashMap 保存统计结果，正常情况下猪只至少一进一出，HashMap 的大小有 ArrayList 的一半即可，以耳号为 key
        HashMap<String, ArrayList<Integer>> logMap = new HashMap<>(hdBdBsBatchRelationLogList.size() / 2);
        String inState = "进";
        String outState = "出";
        ArrayList<String> errorList = new ArrayList<>();
        // 遍历 ArrayList
        for (HdBdBsBatchRelationLog log : hdBdBsBatchRelationLogList) {
            ArrayList<Integer> integers;
            // 如果 HashMap 中已经保存了猪只耳号
            if (logMap.containsKey(log.getBreedingId())) {
                integers = logMap.get(log.getBreedingId());
                // 如果状态是“进”
                if (inState.equals(log.getStateBatch())) {
                    // 将 inNumber 加 1
                    inNumber++;
                    integers.set(0, integers.get(0) + 1);
                } else if (outState.equals(log.getStateBatch())) {
                    // 如果状态是“出”，将 outNumber 加 1
                    outNumber++;
                    integers.set(1, integers.get(1) + 1);
                } else {
                    // 状态为 “FALSE”/空
                    errorNumber++;
                    errorList.add(log.getBreedingId());
                }
                // 添加到 HashMap
                logMap.put(log.getBreedingId(), integers);
            } else {// 如果 HashMap 中还没有保存猪只耳号
                integers = new ArrayList<>(2);
                // 如果状态是“进”
                if (inState.equals(log.getStateBatch())) {
                    // 将记录“进”的值设置为 1
                    inNumber++;
                    integers.add(0, 1);
                    integers.add(1, 0);
                } else if (outState.equals(log.getStateBatch())) {
                    // 如果状态是“出”，将记录“出”的值设置为 1
                    outNumber++;
                    integers.add(0, 0);
                    integers.add(1, 1);
                } else {
                    // 如果状态是“FALSE”/空，将记录“进”、“出”的值都设置为 0
                    errorNumber++;
                    integers.add(0, 0);
                    integers.add(1, 0);
                    // 保存耳号到 errorList
                    errorList.add(log.getBreedingId());
                }
                // 添加到 HashMap
                logMap.put(log.getBreedingId(), integers);
            }
        }
        // 进 - 出 == 1，为管理批次存栏，管理批次待关闭则表示流水有错误
        HashMap<String, ArrayList<Integer>> innerMap = new HashMap<>(hdBdBsBatchRelationLogList.size() / 3);
        // 进 - 出 > 1，流水有错误，数量不多
        HashMap<String, ArrayList<Integer>> inMap = new HashMap<>(8);
        // 进 < 出，流水有错误，数量较少
        HashMap<String, ArrayList<Integer>> outMap = new HashMap<>(8);
        // 进出状态未选择，为 空 或者 “FALSE”，数量极少
        HashMap<String, ArrayList<Integer>> errorMap = new HashMap<>(2);
        // 使用 Map.Entry 遍历 HashMap
        for (Map.Entry<String, ArrayList<Integer>> entry : logMap.entrySet()) {
            // 如果某猪只的进出次数不相等
            if (entry.getValue().get(0) - entry.getValue().get(1) == 1) {
                // 进 - 出 = 1
                in++;
                innerMap.put(entry.getKey(), entry.getValue());
            } else if (entry.getValue().get(0) - entry.getValue().get(1) > 1) {
                // 进 - 出 > 1
                inError++;
                inMap.put(entry.getKey(), entry.getValue());
            } else if (entry.getValue().get(0) < entry.getValue().get(1)) {
                // 进 < 出
                outError++;
                outMap.put(entry.getKey(), entry.getValue());
            } else {
                // 进出次数相等但存在状态为 “FALSE”/空 的数量
                if (errorList.contains(entry.getKey())) {
                    stateError++;
                }
            }
            // 存在状态为 “FALSE”/空 情况的猪只耳号
            if (errorList.contains(entry.getKey())) {
                errorMap.put(entry.getKey(), entry.getValue());
            }
        }
        // 将各个结果分别保存到 ArrayList 中
        ArrayList<HashMap<String, ArrayList<Integer>>> resultList = new ArrayList<>(4);
        resultList.add(0, innerMap);
        resultList.add(1, inMap);
        resultList.add(2, outMap);
        resultList.add(3, errorMap);
        // 所有猪只的检查完成，返回存栏/有问题的猪只耳号
        return resultList;
    }

    /**
     * 格式化输出 HashMap ，每十个元素换行一次
     *
     * @param hashMap 输入 HashMap 类型
     * @return 格式化后的字符串
     */
    private static String hashMapToString(HashMap hashMap) {
        // 使用 HashMap 的迭代器访问其中的元素
        Iterator it = hashMap.entrySet().iterator();
        // HashMap 中没有元素
        if (!it.hasNext()) {
            return "[]";
        }
        // 使用 StringBuilder 保存格式化后的字符串
        StringBuilder sb = new StringBuilder();
        // 遍历 ArrayList
        for (int i = 1; ; i++) {
            // 将一个元素添加到 StringBuilder 中
            sb.append(it.next());
            // 遍历完所有元素，返回字符串
            if (!it.hasNext()) {
                return sb.toString();
            }
            // 每个元素后加上英文逗号和空格
            sb.append(',').append(' ');
            // 每十个元素换行一次
            if (i % 10 == 0) {
                sb.append('\n');
            }
        }
    }

    public static void main(String[] args) {
        // 输入输出文件名
        String fileName = "00127850-梁建婷-KPCC-03-2204212575";
        // 输入文件的路径
        String fileInPath = "C:\\Users\\00137030\\IdeaProjects\\HelloWorld\\src\\log\\logs\\" + fileName;
        // 输出结果文件的路径
        String resultPath = "C:\\Users\\00137030\\IdeaProjects\\HelloWorld\\src\\log\\result\\" + fileName;
        // 输出存栏文件的路径
        String inBatchPath = "C:\\Users\\00137030\\IdeaProjects\\HelloWorld\\src\\compare\\inBatch";
        ArrayList<HdBdBsBatchRelationLog> fileInList = readFile(fileInPath);
        if (!numberCheck(fileInList)) {
            System.out.println("流水编号存在重复，请检查输入数据！");
        }
        // 进行检查并获取返回值
        ArrayList<HashMap<String, ArrayList<Integer>>> checkResult = new ArrayList<>(hdBdBsBatchRelationLogCheck(fileInList));
        // 按自定义格式输出，每 10 个耳号换行
        System.out.println("进：" + inNumber + ", 出：" + outNumber + ", 空：" + errorNumber + '\n' +
                "批次存栏流水有问题的猪只数量为：" + (in + inError + outError + stateError) + "\n\n" +
                "进 - 出 = 1 ：" + in + '\n' +
                hashMapToString(checkResult.get(0)) + "\n\n" +
                "进 - 出 > 1 ：" + inError + '\n' +
                hashMapToString(checkResult.get(1)) + "\n\n" +
                "进 < 出 ：" + outError + '\n' +
                hashMapToString(checkResult.get(2)) + "\n\n" +
                "空 ：" + checkResult.get(3).size() + '\n' +
                hashMapToString(checkResult.get(3))
        );
        // 如果有存栏，将相应耳号写入 compare/inBatch 文件中方便后续与个体存栏进行对比
        if (in > 0) {
            writeFile(checkResult, inBatchPath);
        }
        // 如果有存栏或者流水有错，将相应耳号写入 result/${fileName} 文件中方便后续与个体存栏进行对比
        if ((in + inError + outError + stateError) > 0) {
            writeFile(checkResult, resultPath);
        }
    }
}
